using System;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.Extensions.Logging;
using static SupplementaryAreaPlugin.SupplementaryAreasConstants;

namespace SupplementaryAreaPlugin.Actions;

internal interface IModelRepository
{
    Task<SupplementaryAreasPlugin.SupplementaryModel> GetOrCreateAsync(string recordId, CancellationToken ct);
    Task<SupplementaryAreasPlugin.SupplementaryModel?> LoadAsync(string recordId, CancellationToken ct);
    Task PersistAsync(string recordId, SupplementaryAreasPlugin.SupplementaryModel model, CancellationToken ct);
    bool CheckVersion(SupplementaryAreasPlugin.SupplementaryModel model, int clientVersion, out DPC.APC.Plugins.SDK.PluginResponse? error);
}

internal sealed class ModelRepository : IModelRepository
{
    private readonly SupplementaryAreasPlugin plugin;
    private static readonly JsonSerializerOptions JsonOptions = new(JsonSerializerDefaults.Web) { WriteIndented = false };

    public ModelRepository(SupplementaryAreasPlugin plugin) => this.plugin = plugin;

    public async Task<SupplementaryAreasPlugin.SupplementaryModel> GetOrCreateAsync(string recordId, CancellationToken ct)
    {
        var existing = await LoadAsync(recordId, ct).ConfigureAwait(false);
        if (existing != null)
        {
            existing = EnsureConsistency(recordId, existing, ct);
            return existing;
        }
        var model = new SupplementaryAreasPlugin.SupplementaryModel
        {
            DueDate = null,
            Areas = new System.Collections.Generic.List<SupplementaryAreasPlugin.SupplementaryArea>(),
            Version = 1,
            LastSavedUtc = DateTime.UtcNow
        };
        model.Areas.Add(CreateNewArea(1));
        await PersistAsync(recordId, model, ct).ConfigureAwait(false);
        return model;
    }

    public async Task<SupplementaryAreasPlugin.SupplementaryModel?> LoadAsync(string recordId, CancellationToken ct)
    {
        var app = plugin.App;
        if (app == null || string.IsNullOrWhiteSpace(recordId)) return null;
        string key = ModelKeyPrefix + recordId;
        try
        {
            var json = await app.KeyValue.GetAsync(key, ct).ConfigureAwait(false);
            if (string.IsNullOrWhiteSpace(json)) return null;
            if (json.Length > MaxJsonPayloadSize)
            {
                app.Logger.LogWarning("Model JSON payload too large ({Length}) for {RecordId}", json.Length, recordId);
                return null;
            }
            var model = JsonSerializer.Deserialize<SupplementaryAreasPlugin.SupplementaryModel>(json, JsonOptions);
            if (model == null) return null;
            model.Areas ??= new System.Collections.Generic.List<SupplementaryAreasPlugin.SupplementaryArea>();
            return model;
        }
        catch (OperationCanceledException) { throw; }
        catch (Exception ex)
        {
            app.Logger.LogWarning(ex, "Load model failed for {RecordId}", recordId);
            return null;
        }
    }

    public async Task PersistAsync(string recordId, SupplementaryAreasPlugin.SupplementaryModel model, CancellationToken ct)
    {
        var app = plugin.App;
        if (app == null) return;
        string key = ModelKeyPrefix + recordId;
        try
        {
            var json = JsonSerializer.Serialize(model, JsonOptions);
            
            
            try { await app.KeyValue.DeleteAsync(key, ct).ConfigureAwait(false); } catch { }
            await app.KeyValue.SetAsync(key, json, null, ct).ConfigureAwait(false);
            if (app.Logger.IsEnabled(LogLevel.Trace))
                app.Logger.LogTrace("Persisted model {RecordId} v{Version}", recordId, model.Version);
        }
        catch (OperationCanceledException) { throw; }
        catch (Exception ex)
        {
            app.Logger.LogError(ex, "Persist model failed for {RecordId}", recordId);
            throw;
        }
    }

    public bool CheckVersion(SupplementaryAreasPlugin.SupplementaryModel model, int clientVersion, out DPC.APC.Plugins.SDK.PluginResponse? error)
    {
        error = null;
        if (clientVersion != model.Version)
        {
            error = SupplementaryAreasPlugin.CreateError(400, "concurrency_conflict");
            return false;
        }
        return true;
    }

    private SupplementaryAreasPlugin.SupplementaryModel EnsureConsistency(string recordId, SupplementaryAreasPlugin.SupplementaryModel model, CancellationToken ct)
    {
        model.Areas ??= new System.Collections.Generic.List<SupplementaryAreasPlugin.SupplementaryArea>();
        model.Version = Math.Max(1, model.Version);
        if (model.Areas.Count == 0)
        {
            model.Areas.Add(CreateNewArea(1));
            model.Version++;
            model.LastSavedUtc = DateTime.UtcNow;
            PersistAsync(recordId, model, ct).GetAwaiter().GetResult();
        }
        return model;
    }

    private static SupplementaryAreasPlugin.SupplementaryArea CreateNewArea(int index) => new()
    {
        Id = Guid.NewGuid().ToString("n"),
        Index = index,
        Order = index,
        Label = null,
        ExecutiveDirectors = new System.Collections.Generic.List<SupplementaryAreasPlugin.Person>(),
        Allocators = new System.Collections.Generic.List<SupplementaryAreasPlugin.Person>(),
        Contributors = new System.Collections.Generic.List<SupplementaryAreasPlugin.Person>(),
        Comments = new System.Collections.Generic.List<SupplementaryAreasPlugin.Comment>(),
        EndorsementComplete = false
    };
}
