using System;
using System.Collections.Generic;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using DPC.APC.Plugins.SDK;
using Microsoft.Extensions.Logging;
using static SupplementaryAreaPlugin.SupplementaryAreasConstants;

// Encapsulates the people search logic previously embedded in SupplementaryAreasPlugin.
namespace SupplementaryAreaPlugin.Actions;

internal interface IPeopleSearchContext
{
    IAppAccess? App { get; }
    PluginResponse Ok(object shape);
    PluginResponse Fail(int status, string code, string? detail = null);
    string GetQueryValue(IReadOnlyDictionary<string,string?> query, string key);
    bool IsValidStringLength(string? value, int maxLength);
}

internal sealed class ActionPeopleSearch
{
    private readonly IPeopleSearchContext ctx;

    // Constants to avoid magic strings & typos
    private const string PropUsers = "users";
    private const string PropError = "error";
    private const string PropGraphUserId = "GraphUserId";

    public ActionPeopleSearch(IPeopleSearchContext ctx) => this.ctx = ctx;

    // Primary (async) execution entry point
    public async Task<PluginResponse> ExecuteAsync(
        IReadOnlyDictionary<string, string?> query,
        CancellationToken ct = default)
    {
        string term = ctx.GetQueryValue(query, QuerySearchTerm)?.Trim() ?? string.Empty;

        if (!ctx.IsValidStringLength(term, MaxSearchTermLength))
            return ctx.Fail(400, "search_term_too_long", $"Search term exceeds maximum length of {MaxSearchTermLength} characters");

        if (term.Length < 2)
            return OkEmpty();

        if (string.IsNullOrWhiteSpace(term))
            return OkEmpty();

        var app = ctx.App;
        if (app == null)
            return ctx.Fail(500, ErrorNotInitialised);

        try
        {
            var payload = BuildPayload(term);
            if (app.Logger.IsEnabled(LogLevel.Trace))
                app.Logger.LogTrace("peopleSearch: invoking {Action} termLen={Len}", UniversalActionSearchUsers, term.Length);

            string raw = await app.UniversalAppAccess
                .InvokeActionAsync(UniversalActionSearchUsers, payload, ct)
                .ConfigureAwait(false);

            if (string.IsNullOrWhiteSpace(raw))
                return OkEmpty();

            // Parse & shape response within document lifetime
            using var doc = JsonDocument.Parse(raw);
            var root = doc.RootElement;

            if (root.TryGetProperty(PropError, out var errEl))
            {
                var err = errEl.GetString();
                return ctx.Ok(new { results = Array.Empty<object>(), error = err });
            }

            var results = ExtractUsers(root);
            if (results.Count == 0)
                return OkEmpty();
            return ctx.Ok(new { results });
        }
        catch (OperationCanceledException)
        {
            return ctx.Fail(499, "request_cancelled");
        }
        catch (Exception ex)
        {
            app?.Logger.LogWarning(ex, "PeopleSearch action failed for term {Term}", term);
            // Maintain prior contract: still 200 with soft error
            return ctx.Ok(new { results = Array.Empty<object>(), error = "action_failed" });
        }
    }

    // Backwards-compatible sync wrapper (avoid widespread signature changes immediately)
    public PluginResponse Execute(IReadOnlyDictionary<string, string?> query)
        => ExecuteAsync(query, CancellationToken.None).GetAwaiter().GetResult();

    private PluginResponse OkEmpty() => ctx.Ok(new { results = Array.Empty<object>() });

    private static string BuildPayload(string term)
        => JsonSerializer.Serialize(new { searchTerm = term, top = PeopleSearchLimit });

    private List<object> ExtractUsers(JsonElement root)
    {
        var list = new List<object>();
        if (!root.TryGetProperty(PropUsers, out var usersEl) || usersEl.ValueKind != JsonValueKind.Array)
            return list;

        foreach (var u in usersEl.EnumerateArray())
        {
            string? graphUserId = u.TryGetProperty(PropGraphUserId, out var gid) ? gid.GetString() : null;
            if (string.IsNullOrWhiteSpace(graphUserId))
                continue;
            list.Add(u); // Serialize immediately before JsonDocument disposal
            if (list.Count >= PeopleSearchLimit) // defensive cap (should match server limit)
                break;
        }
        return list;
    }
}
