using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using DPC.APC.Plugins.SDK;
using Microsoft.Extensions.Logging;
using static SupplementaryAreaPlugin.SupplementaryAreasConstants;

namespace SupplementaryAreaPlugin.Actions;

internal interface IUpdateRecordContext
{
    IAppAccess? App { get; }
    PluginResponse Ok(object shape);
    PluginResponse Fail(int status, string code, string? detail = null);
}

internal sealed class ActionUpdateRecord
{
    private readonly IUpdateRecordContext ctx;
    private readonly IModelRepository repo;

    public ActionUpdateRecord(IUpdateRecordContext ctx, IModelRepository repo)
    {
        this.ctx = ctx;
        this.repo = repo;
    }

    public async Task<PluginResponse> ExecuteAsync(string recordId, JsonElement root, CancellationToken ct)
    {
        var app = ctx.App;
        if (app == null) return ctx.Fail(500, ErrorNotInitialised);

        // 1. Validate payload structure
        if (root.ValueKind != JsonValueKind.Object)
            return ctx.Fail(400, ErrorInvalidModel, "Payload must be a JSON object");

        // 2. Load existing model for concurrency check
        var existingModel = await repo.GetOrCreateAsync(recordId, ct).ConfigureAwait(false);

        // 3. Parse and validate incoming complete state
        var parseResult = await ParseCompleteStateAsync(root, existingModel, app.Logger, ct).ConfigureAwait(false);
        if (!parseResult.IsValid)
            return ctx.Fail(400, parseResult.ErrorCode!, parseResult.ErrorDetail);

        var incomingState = parseResult.State!;

        // 4. Perform optimistic concurrency check
        if (!repo.CheckVersion(existingModel, incomingState.Version, out var versionErr))
        {
            return versionErr!;
        }

        // 5. Clone existing model for change tracking
        var oldModelSnapshot = CloneModel(existingModel);

        // 6. Update model with complete state
        existingModel.DueDate = incomingState.DueDate;
        existingModel.Areas = incomingState.Areas;

        // 7. Track changes between old and new models
        var changeTracker = new SupplementaryChangeTracker(oldModelSnapshot, existingModel, app.Logger);
        var changes = changeTracker.DetectChanges();

        // 8. Increment version and persist to plugin data
        var previousVersion = existingModel.Version;
        existingModel.Version++;
        existingModel.LastSavedUtc = DateTime.UtcNow;
        
        app.Logger.LogDebug("Saving model {RecordId} v{PrevVersion}->{NewVersion} with {AreaCount} areas",
            recordId, previousVersion, existingModel.Version, existingModel.Areas.Count);

        await repo.PersistAsync(recordId, existingModel, ct).ConfigureAwait(false);

        // 9. Extract field values and filter by active content type fields
        var fieldUpdates = ExtractFieldValuesFromModel(existingModel);
        app.Logger.LogDebug("Extracted {FieldCount} field updates for record fields synchronization", fieldUpdates.Count);

        // 10. Filter updates by active fields (removes fields that don't exist in content type)
        var filteredUpdates = await FilterUpdatesByActiveFieldsAsync(recordId, root, fieldUpdates, ct, app).ConfigureAwait(false);
        if (filteredUpdates.Count == 0)
        {
            app.Logger.LogWarning("No active fields to update for {RecordId} - all fields filtered out", recordId);
            // Still return success as model is saved, just no record sync
            // Process change notifications
            await ProcessChangeEventAsync(recordId, changes, ct).ConfigureAwait(false);
            return ctx.Ok(new { model = existingModel });
        }

        // 11. Synchronize filtered updates to record
        var syncResult = await SynchronizeToRecordFields(recordId, filteredUpdates, ct).ConfigureAwait(false);
        if (!syncResult.IsSuccess)
            return syncResult.ErrorResponse!;

        // 12. Process change notifications and audit logs
        await ProcessChangeEventAsync(recordId, changes, ct);

        // 13. Return updated model
        return ctx.Ok(new { model = existingModel });
    }

    private async Task<(bool IsValid, string? ErrorCode, string? ErrorDetail, CompleteStatePayload? State)> ParseCompleteStateAsync(
        JsonElement root,
        SupplementaryAreasPlugin.SupplementaryModel existingModel,
        ILogger logger,
        CancellationToken ct)
    {
        try
        {
            // Parse version
            int version = root.TryGetProperty("version", out var vEl) && vEl.TryGetInt32(out var v) ? v : 1;

            // Parse dueDate and convert to ISO 8601 format string
            string? dueDate = null;
            if (root.TryGetProperty(JsonDueDate, out var ddEl))
            {
                if (ddEl.ValueKind == JsonValueKind.String)
                {
                    var dateString = ddEl.GetString();
                    if (!string.IsNullOrWhiteSpace(dateString))
                    {
                        dueDate = ParseAndFormatDateString(dateString);
                    }
                }
            }

            // Parse areas array
            var areas = new List<SupplementaryAreasPlugin.SupplementaryArea>();
            if (root.TryGetProperty(JsonAreas, out var areasEl) && areasEl.ValueKind == JsonValueKind.Array)
            {
                if (areasEl.GetArrayLength() > MaxAreasLimit)
                {
                    return (false, "max_areas", $"Maximum {MaxAreasLimit} areas allowed", null);
                }

                foreach (var areaEl in areasEl.EnumerateArray())
                {
                    if (areaEl.ValueKind != JsonValueKind.Object) continue;

                    var area = await ParseAreaAsync(areaEl, existingModel, logger, ct).ConfigureAwait(false);
                    if (area != null)
                    {
                        areas.Add(area);
                    }
                }
            }

            // Ensure at least one area
            if (areas.Count == 0)
            {
                areas.Add(CreateNewArea(1));
            }

            // Reorder and reindex areas
            areas = ReorderAndReindexAreas(areas);

            return (true, null, null, new CompleteStatePayload
            {
                Version = version,
                DueDate = dueDate,
                Areas = areas
            });
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Failed to parse complete state payload");
            return (false, ErrorInvalidJson, ex.Message, null);
        }
    }

    private async Task<SupplementaryAreasPlugin.SupplementaryArea?> ParseAreaAsync(
        JsonElement areaEl, 
        SupplementaryAreasPlugin.SupplementaryModel existingModel,
        ILogger logger,
        CancellationToken ct)
    {
        try
        {
            var id = areaEl.TryGetProperty("id", out var idEl) && idEl.ValueKind == JsonValueKind.String
                ? idEl.GetString()
                : Guid.NewGuid().ToString("n");

            // Find existing area to preserve comments
            var existingArea = existingModel.Areas?.FirstOrDefault(a => string.Equals(a.Id, id, StringComparison.OrdinalIgnoreCase));

            var label = areaEl.TryGetProperty("label", out var labelEl) && labelEl.ValueKind == JsonValueKind.String
                ? ValidateAndTruncate(labelEl.GetString(), MaxLabelLength)
                : null;

            var order = areaEl.TryGetProperty("order", out var orderEl) && orderEl.TryGetInt32(out var o)
                ? Math.Max(0, Math.Min(1000, o))
                : 0;

            var index = areaEl.TryGetProperty("index", out var indexEl) && indexEl.TryGetInt32(out var i)
                ? Math.Max(0, Math.Min(1000, i))
                : order;

            var executiveDirectors = ParsePersonArray(areaEl, "executiveDirectors");
            var allocators = ParsePersonArray(areaEl, "allocators");
            var contributors = ParsePersonArray(areaEl, "contributors");

            // Parse existing comments from payload
            var comments = ParseCommentArray(areaEl, "comments");
            
            // If no comments array provided but existing area has comments, preserve them
            // Create a new list to avoid reference issues
            if (comments.Count == 0 && existingArea != null && existingArea.Comments != null && existingArea.Comments.Count > 0)
            {
                comments = new List<SupplementaryAreasPlugin.Comment>(existingArea.Comments);
            }

            // Check for new comment to add
            if (areaEl.TryGetProperty("newComment", out var newCommentEl) && newCommentEl.ValueKind == JsonValueKind.String)
            {
                var newCommentText = newCommentEl.GetString();
                if (!string.IsNullOrWhiteSpace(newCommentText))
                {
                    var currentUser = await GetCurrentUserAsync(ct).ConfigureAwait(false);
                    var now = DateTime.UtcNow;
                    var newComment = new SupplementaryAreasPlugin.Comment
                    {
                        Id = Guid.NewGuid().ToString("n"),
                        CreatedUtc = now,
                        ModifiedUtc = now,
                        Author = currentUser,
                        Text = ValidateAndTruncate(newCommentText, MaxCommentsLength) ?? string.Empty
                    };
                    comments.Add(newComment);
                    
                    logger.LogDebug("Added new comment to area {AreaId}: {CommentText}", 
                        id ?? "unknown", newCommentText.Substring(0, Math.Min(50, newCommentText.Length)));
                }
            }

            var endorsementComplete = areaEl.TryGetProperty("endorsementComplete", out var ecEl) &&
                                      (ecEl.ValueKind == JsonValueKind.True ||
                                       (ecEl.ValueKind == JsonValueKind.String && ecEl.GetString() == "true"));

            return new SupplementaryAreasPlugin.SupplementaryArea
            {
                Id = id ?? Guid.NewGuid().ToString("n"),
                Index = index,
                Order = order,
                Label = label,
                ExecutiveDirectors = executiveDirectors,
                Allocators = allocators,
                Contributors = contributors,
                Comments = comments,
                EndorsementComplete = endorsementComplete
            };
        }
        catch (Exception ex)
        {
            logger.LogWarning(ex, "Failed to parse area element, skipping");
            return null;
        }
    }

    private List<SupplementaryAreasPlugin.Person> ParsePersonArray(JsonElement parent, string propertyName)
    {
        var list = new List<SupplementaryAreasPlugin.Person>();
        if (!parent.TryGetProperty(propertyName, out var arrayEl)) return list;

        if (arrayEl.ValueKind == JsonValueKind.Array)
        {
            foreach (var item in arrayEl.EnumerateArray())
            {
                if (item.ValueKind == JsonValueKind.Object)
                {
                    var person = ParsePersonObject(item);
                    if (person != null)
                    {
                        list.Add(person);
                    }
                }
            }
        }

        return list;
    }

    private List<SupplementaryAreasPlugin.Comment> ParseCommentArray(JsonElement parent, string propertyName)
    {
        var list = new List<SupplementaryAreasPlugin.Comment>();
        if (!parent.TryGetProperty(propertyName, out var arrayEl)) return list;

        if (arrayEl.ValueKind == JsonValueKind.Array)
        {
            foreach (var item in arrayEl.EnumerateArray())
            {
                if (item.ValueKind == JsonValueKind.Object)
                {
                    var comment = ParseCommentObject(item);
                    if (comment != null)
                    {
                        list.Add(comment);
                    }
                }
            }
        }

        return list;
    }

    private SupplementaryAreasPlugin.Comment? ParseCommentObject(JsonElement commentEl)
    {
        try
        {
            var id = GetStringProperty(commentEl, "id");
            if (string.IsNullOrWhiteSpace(id))
            {
                id = Guid.NewGuid().ToString("n");
            }

            var text = GetStringProperty(commentEl, "text");
            if (string.IsNullOrWhiteSpace(text)) return null; // Comment must have text

            DateTime createdUtc = DateTime.UtcNow;
            if (commentEl.TryGetProperty("createdUtc", out var dateEl) && dateEl.ValueKind == JsonValueKind.String)
            {
                if (DateTime.TryParse(dateEl.GetString(), out var parsedDate))
                {
                    createdUtc = parsedDate;
                }
            }

            DateTime modifiedUtc = createdUtc;
            if (commentEl.TryGetProperty("modifiedUtc", out var modDateEl) && modDateEl.ValueKind == JsonValueKind.String)
            {
                if (DateTime.TryParse(modDateEl.GetString(), out var parsedModDate))
                {
                    modifiedUtc = parsedModDate;
                }
            }

            SupplementaryAreasPlugin.Person? author = null;
            if (commentEl.TryGetProperty("author", out var authorEl) && authorEl.ValueKind == JsonValueKind.Object)
            {
                author = ParsePersonObject(authorEl);
            }

            return new SupplementaryAreasPlugin.Comment
            {
                Id = id,
                CreatedUtc = createdUtc,
                ModifiedUtc = modifiedUtc,
                Author = author,
                Text = ValidateAndTruncate(text, MaxCommentsLength) ?? string.Empty
            };
        }
        catch (Exception ex)
        {
            return null;
        }
    }

    private SupplementaryAreasPlugin.Person? ParsePersonObject(JsonElement personEl)
    {
        // Extract properties from person object - support both PascalCase and camelCase
        var person = new SupplementaryAreasPlugin.Person
        {
            // Try PascalCase first, then camelCase, then "key" for GraphUserId
            GraphUserId = GetStringProperty(personEl, "GraphUserId") 
                ?? GetStringProperty(personEl, "graphUserId"),
            
            DisplayName = GetStringProperty(personEl, "DisplayName") 
                ?? GetStringProperty(personEl, "displayName"),
            
            JobTitle = GetStringProperty(personEl, "JobTitle") 
                ?? GetStringProperty(personEl, "jobTitle"),
            
            Mail = GetStringProperty(personEl, "Mail") 
                ?? GetStringProperty(personEl, "mail"),
            
            UserPrincipalName = GetStringProperty(personEl, "UserPrincipalName") 
                ?? GetStringProperty(personEl, "userPrincipalName")
        };

        // Validate that we have at least one identifier
        if (string.IsNullOrWhiteSpace(person.GraphUserId) && 
            string.IsNullOrWhiteSpace(person.UserPrincipalName) && 
            string.IsNullOrWhiteSpace(person.Mail))
        {
            return null; // Invalid person - no identifier
        }

        return person;
    }

    private string? GetStringProperty(JsonElement element, string propertyName)
    {
        if (element.TryGetProperty(propertyName, out var prop) && prop.ValueKind == JsonValueKind.String)
        {
            return prop.GetString();
        }
        return null;
    }

    private async Task<SupplementaryAreasPlugin.Person?> GetCurrentUserAsync(CancellationToken ct)
    {
        var app = ctx.App;
        if (app == null) return null;

        try
        {
            string json = await app.UniversalAppAccess.InvokeActionAsync(UniversalActionGetCurrentUser, string.Empty, ct).ConfigureAwait(false);
            if (string.IsNullOrWhiteSpace(json)) return null;

            using var doc = JsonDocument.Parse(json);
            var root = doc.RootElement;

            if (root.ValueKind != JsonValueKind.Object) return null;

            return new SupplementaryAreasPlugin.Person
            {
                GraphUserId = GetStringProperty(root, "GraphUserId") ?? GetStringProperty(root, "id"),
                DisplayName = GetStringProperty(root, "DisplayName") ?? GetStringProperty(root, "displayName"),
                JobTitle = GetStringProperty(root, "JobTitle") ?? GetStringProperty(root, "jobTitle"),
                Mail = GetStringProperty(root, "Mail") ?? GetStringProperty(root, "mail"),
                UserPrincipalName = GetStringProperty(root, "UserPrincipalName") ?? GetStringProperty(root, "userPrincipalName")
            };
        }
        catch (Exception ex)
        {
            app.Logger.LogWarning(ex, "Failed to get current user for comment author");
            return null;
        }
    }

    private List<SupplementaryAreasPlugin.SupplementaryArea> ReorderAndReindexAreas(
        List<SupplementaryAreasPlugin.SupplementaryArea> areas)
    {
        areas = areas.OrderBy(a => a.Order).ThenBy(a => a.Id).ToList();
        for (int i = 0; i < areas.Count; i++)
        {
            var a = areas[i];
            a.Index = i + 1;
            if (a.Order <= 0) a.Order = a.Index;
        }
        return areas;
    }

    private List<(string key, object? value)> ExtractFieldValuesFromModel(SupplementaryAreasPlugin.SupplementaryModel model)
    {
        var updates = new List<(string key, object? value)>();

        // Due date field
        updates.Add((SupplementaryAreasPlugin.SupplementaryModel.DueDateFieldInternalName, model.DueDate));

        // Aggregate distinct person objects from all areas for SharePoint people picker fields
        // Use dictionary with Key as key to deduplicate while preserving complete Person objects
        var allExecutiveDirectors = new Dictionary<string, SupplementaryAreasPlugin.Person>(StringComparer.OrdinalIgnoreCase);
        var allAllocators = new Dictionary<string, SupplementaryAreasPlugin.Person>(StringComparer.OrdinalIgnoreCase);
        var allContributors = new Dictionary<string, SupplementaryAreasPlugin.Person>(StringComparer.OrdinalIgnoreCase);

        foreach (var area in model.Areas)
        {
            if (area.ExecutiveDirectors != null)
            {
                foreach (var person in area.ExecutiveDirectors)
                {
                    var personId = ExtractIdFromPerson(person);
                    if (!string.IsNullOrWhiteSpace(personId))
                    {
                        // Store complete person object, deduplicated by ID
                        allExecutiveDirectors[personId] = person;
                    }
                }
            }

            if (area.Allocators != null)
            {
                foreach (var person in area.Allocators)
                {
                    var personId = ExtractIdFromPerson(person);
                    if (!string.IsNullOrWhiteSpace(personId))
                    {
                        allAllocators[personId] = person;
                    }
                }
            }

            if (area.Contributors != null)
            {
                foreach (var person in area.Contributors)
                {
                    var personId = ExtractIdFromPerson(person);
                    if (!string.IsNullOrWhiteSpace(personId))
                    {
                        allContributors[personId] = person;
                    }
                }
            }
        }

        // Add aggregated person object lists to updates for SharePoint people picker fields
        updates.Add((SupplementaryAreasPlugin.SupplementaryModel.ExecutiveDirectorsFieldInternalName, 
            allExecutiveDirectors.Count > 0 ? allExecutiveDirectors.Values.ToList() : new List<SupplementaryAreasPlugin.Person>()));
        
        updates.Add((SupplementaryAreasPlugin.SupplementaryModel.AllocatorsFieldInternalName, 
            allAllocators.Count > 0 ? allAllocators.Values.ToList() : new List<SupplementaryAreasPlugin.Person>()));
        
        updates.Add((SupplementaryAreasPlugin.SupplementaryModel.ContributorsFieldInternalName, 
            allContributors.Count > 0 ? allContributors.Values.ToList() : new List<SupplementaryAreasPlugin.Person>()));

        return updates;
    }

    private string? ExtractIdFromPerson(SupplementaryAreasPlugin.Person person)
    {
        if (person == null) return null;
        
        // Prefer GraphUserId
        if (!string.IsNullOrWhiteSpace(person.GraphUserId)) return person.GraphUserId;
        
        // Then UserPrincipalName
        if (!string.IsNullOrWhiteSpace(person.UserPrincipalName)) return person.UserPrincipalName;
        
        // Finally Mail
        if (!string.IsNullOrWhiteSpace(person.Mail)) return person.Mail;
        
        return null;
    }

    private async Task<List<(string key, object? value)>> FilterUpdatesByActiveFieldsAsync(
        string recordId,
        JsonElement root,
        List<(string key, object? value)> updates,
        CancellationToken ct,
        IAppAccess app)
    {
        if (updates.Count == 0) return updates;

        try
        {
            // Get recordType from payload or fetch the record
            string? recordType = null;
            if (root.TryGetProperty("recordType", out var rtEl) && rtEl.ValueKind == JsonValueKind.String)
                recordType = rtEl.GetString();

            if (string.IsNullOrWhiteSpace(recordType))
            {
                var summary = await app.Records.GetRecordAsync(recordId, ct).ConfigureAwait(false);
                recordType = summary?.Type;
            }

            if (string.IsNullOrWhiteSpace(recordType))
            {
                app.Logger.LogWarning("Cannot filter fields for {RecordId} - recordType not available", recordId);
                return updates; // Fail open - return all updates
            }

            // Fetch active content type fields
            var activeFields = await FetchActiveFields(recordType, ct, app).ConfigureAwait(false);
            if (activeFields.Count == 0)
            {
                app.Logger.LogWarning("No active fields found for recordType {RecordType}", recordType);
                return updates; // Fail open
            }

            // Filter updates to only include active fields
            var allowed = new HashSet<string>(activeFields.Select(f => f.InternalName), StringComparer.OrdinalIgnoreCase);
            var filtered = new List<(string key, object? value)>();
            var missing = new List<string>();

            foreach (var update in updates)
            {
                if (allowed.Contains(update.key))
                {
                    filtered.Add(update);
                }
                else
                {
                    missing.Add(update.key);
                }
            }

            if (missing.Count > 0)
            {
                app.Logger.LogInformation(
                    "Filtered out {MissingCount} missing field(s) for {RecordId} ({RecordType}): {Fields}",
                    missing.Count,
                    recordId,
                    recordType,
                    string.Join(", ", missing.Take(10)) + (missing.Count > 10 ? "..." : string.Empty));
            }

            return filtered;
        }
        catch (Exception ex)
        {
            app.Logger.LogWarning(ex, "Error filtering fields for {RecordId} - proceeding with all updates", recordId);
            return updates; // Fail open on error
        }
    }

    private async Task<List<FieldDescriptorLite>> FetchActiveFields(
        string recordType,
        CancellationToken ct,
        IAppAccess app)
    {
        var list = new List<FieldDescriptorLite>();
        try
        {
            string parameters = JsonSerializer.Serialize(new { contentType = recordType });
            string json = await app.UniversalAppAccess.InvokeActionAsync(UniversalActionGetContentTypeFields, parameters, ct).ConfigureAwait(false);

            using var doc = JsonDocument.Parse(json);
            if (doc.RootElement.ValueKind == JsonValueKind.Array)
            {
                foreach (var el in doc.RootElement.EnumerateArray())
                {
                    ct.ThrowIfCancellationRequested();
                    if (!el.TryGetProperty("internalName", out var nameEl)) continue;

                    string? internalName = nameEl.ValueKind == JsonValueKind.String
                        ? nameEl.GetString()
                        : (nameEl.ValueKind == JsonValueKind.Number ? nameEl.GetRawText() : null);

                    if (string.IsNullOrWhiteSpace(internalName)) continue;

                    list.Add(new FieldDescriptorLite { InternalName = internalName! });
                }
            }
        }
        catch (Exception ex)
        {
            app.Logger.LogWarning(ex, "Failed fetching active fields for recordType {RecordType}", recordType);
        }
        return list;
    }

    private async Task<(bool IsSuccess, PluginResponse? ErrorResponse)> SynchronizeToRecordFields(
        string recordId,
        List<(string key, object? value)> updates,
        CancellationToken ct)
    {
        var app = ctx.App!;
        try
        {
            // Build parameters JSON for UpdateRecordFields action
            var parametersJson = BuildUpdateRecordFieldsParameters(recordId, updates);

            app.Logger.LogTrace("Invoking {Action} for {RecordId} with {FieldCount} fields",
                UniversalActionUpdateRecordFields, recordId, updates.Count);

            // Invoke record update action
            var response = await app.UniversalAppAccess.InvokeActionAsync(
                UniversalActionUpdateRecordFields,
                parametersJson,
                ct).ConfigureAwait(false);

            // Check for errors in response
            var error = ParseActionError(response, app.Logger);
            if (error != null)
                return (false, error);

            app.Logger.LogDebug("Successfully synchronized {FieldCount} fields to record for {RecordId}",
                updates.Count, recordId);

            return (true, null);
        }
        catch (OperationCanceledException)
        {
            throw;
        }
        catch (Exception ex)
        {
            app.Logger.LogError(ex, "Failed to synchronize fields to record for {RecordId}", recordId);
            return (false, SupplementaryAreasPlugin.CreateError(500, ErrorActionFailed,
                "Failed to synchronize fields to record"));
        }
    }

    private string BuildUpdateRecordFieldsParameters(string recordId, List<(string key, object? value)> updates)
    {
        using var ms = new System.IO.MemoryStream();
        using var writer = new Utf8JsonWriter(ms);

        writer.WriteStartObject();
        writer.WriteString("recordId", recordId);
        writer.WritePropertyName("customFields");
        writer.WriteStartObject();

        foreach (var (key, value) in updates)
        {
            writer.WritePropertyName(key);
            WriteFieldValue(writer, value);
        }

        writer.WriteEndObject();
        writer.WriteEndObject();
        writer.Flush();

        return System.Text.Encoding.UTF8.GetString(ms.ToArray());
    }

    private void WriteFieldValue(Utf8JsonWriter writer, object? value)
    {
        if (value == null)
        {
            writer.WriteNullValue();
            return;
        }

        switch (value)
        {
            case string s:
                writer.WriteStringValue(s);
                break;
            case bool b:
                writer.WriteBooleanValue(b);
                break;
            case int i:
                writer.WriteNumberValue(i);
                break;
            case List<string> stringList:
                writer.WriteStartArray();
                foreach (var item in stringList)
                {
                    writer.WriteStringValue(item);
                }
                writer.WriteEndArray();
                break;
            case List<SupplementaryAreasPlugin.Person> personList:
                // Serialize Person objects as JSON array for SharePoint people picker fields
                writer.WriteStartArray();
                foreach (var person in personList)
                {
                    var personJson = JsonSerializer.Serialize(person);
                    writer.WriteRawValue(personJson);
                }
                writer.WriteEndArray();
                break;
            default:
                // Serialize as JSON for any other type
                var json = JsonSerializer.Serialize(value);
                writer.WriteRawValue(json);
                break;
        }
    }

    private PluginResponse? ParseActionError(string raw, ILogger logger)
    {
        try
        {
            using var doc = JsonDocument.Parse(raw);
            var root = doc.RootElement;
            if (root.ValueKind == JsonValueKind.Object && root.TryGetProperty(JsonError, out var errProp))
            {
                var code = errProp.GetString();
                return MapError(code);
            }
        }
        catch (Exception ex)
        {
            logger.LogWarning(ex, "Could not parse action response for {Action}", UniversalActionUpdateRecordFields);
        }
        return null;
    }

    private PluginResponse MapError(string? code)
        => code switch
        {
            "missing_parameters" or "missing_recordId" or "missing_customFields" or "invalid_json" or
            "max_areas" or "invalid_model" or "no_updates" =>
                SupplementaryAreasPlugin.CreateError(400, code!),
            "unauthorised" or "forbidden" =>
                SupplementaryAreasPlugin.CreateError(403, code!),
            "concurrency_conflict" =>
                SupplementaryAreasPlugin.CreateError(409, code!, "Model has been modified by another user"),
            _ =>
                SupplementaryAreasPlugin.CreateError(500, code ?? ErrorUpdateFailed)
        };

    private SupplementaryAreasPlugin.SupplementaryModel CloneModel(SupplementaryAreasPlugin.SupplementaryModel source)
    {
        // Deep clone the model for change tracking
        return new SupplementaryAreasPlugin.SupplementaryModel
        {
            DueDate = source.DueDate,
            Version = source.Version,
            LastSavedUtc = source.LastSavedUtc,
            Areas = source.Areas?.Select(area => new SupplementaryAreasPlugin.SupplementaryArea
            {
                Id = area.Id,
                Index = area.Index,
                Order = area.Order,
                Label = area.Label,
                ExecutiveDirectors = ClonePersonList(area.ExecutiveDirectors),
                Allocators = ClonePersonList(area.Allocators),
                Contributors = ClonePersonList(area.Contributors),
                Comments = CloneCommentList(area.Comments),
                EndorsementComplete = area.EndorsementComplete
            }).ToList() ?? new List<SupplementaryAreasPlugin.SupplementaryArea>()
        };
    }

    private List<SupplementaryAreasPlugin.Person> ClonePersonList(List<SupplementaryAreasPlugin.Person>? source)
    {
        if (source == null) return new List<SupplementaryAreasPlugin.Person>();
        
        return source.Select(p => new SupplementaryAreasPlugin.Person
        {
            GraphUserId = p.GraphUserId,
            DisplayName = p.DisplayName,
            JobTitle = p.JobTitle,
            Mail = p.Mail,
            UserPrincipalName = p.UserPrincipalName
        }).ToList();
    }

    private List<SupplementaryAreasPlugin.Comment> CloneCommentList(List<SupplementaryAreasPlugin.Comment>? source)
    {
        if (source == null) return new List<SupplementaryAreasPlugin.Comment>();
        
        return source.Select(c => new SupplementaryAreasPlugin.Comment
        {
            Id = c.Id,
            Text = c.Text,
            CreatedUtc = c.CreatedUtc,
            ModifiedUtc = c.ModifiedUtc,
            Author = c.Author == null ? null : new SupplementaryAreasPlugin.Person
            {
                GraphUserId = c.Author.GraphUserId,
                DisplayName = c.Author.DisplayName,
                JobTitle = c.Author.JobTitle,
                Mail = c.Author.Mail,
                UserPrincipalName = c.Author.UserPrincipalName
            }
        }).ToList();
    }

    private async Task ProcessChangeEventAsync(
        string recordId,
        ModelChanges changes,
        CancellationToken ct)
    {
        var app = ctx.App;
        if (app == null) return;

        try
        {
            // Get current user for audit logs and notifications
            var currentUser = await GetCurrentUserAsync(ct).ConfigureAwait(false);

            // Audit logs
            await SaveAuditLogEntriesAsync(recordId, changes.AuditLogEntries, currentUser, ct)
                .ConfigureAwait(false);

            // Send notifications
            var notificationService = new SupplementaryNotificationService(app, app.Logger);
            await notificationService.SendNotificationsAsync(recordId, changes, ct)
                .ConfigureAwait(false);
        }
        catch (Exception ex)
        {
            app.Logger.LogError(ex, "Failed to process change notifications for record {RecordId}", recordId);
            // Don't throw - notifications are best-effort
        }
    }

    private async Task SaveAuditLogEntriesAsync(
        string recordId,
        List<AuditLogEntry> entries,
        SupplementaryAreasPlugin.Person? currentUser,
        CancellationToken ct)
    {
        var app = ctx.App;
        if (app == null || entries.Count == 0) return;

        app.Logger.LogInformation("Saving {Count} audit log entries for record {RecordId}",
            entries.Count, recordId);

        foreach (var entry in entries)
        {
            try
            {
                await SaveAuditLogEntryAsync(recordId, entry, currentUser, ct)
                    .ConfigureAwait(false);
            }
            catch (Exception ex)
            {
                app.Logger.LogWarning(ex,
                    "Failed to save audit log entry for field {Field}",
                    entry.Field);
            }
        }
    }

    private async Task SaveAuditLogEntryAsync(
        string recordId,
        AuditLogEntry entry,
        SupplementaryAreasPlugin.Person? currentUser,
        CancellationToken ct)
    {
        var app = ctx.App;
        if (app == null) return;

        // Build statement based on area information
        var statement = BuildAuditStatement(entry, currentUser);

        var parameters = new
        {
            recordId,
            @event = "Supplementary Area Updated",
            statement,
            field = entry.Field,
            oldValue = entry.OldValue,
            newValue = entry.NewValue
        };

        var parametersJson = JsonSerializer.Serialize(parameters);

        app.Logger.LogDebug("Saving audit log entry: {Params}", parametersJson);

        var response = await app.UniversalAppAccess.InvokeActionAsync(
            UniversalActionLogAuditEntry,
            parametersJson,
            ct).ConfigureAwait(false);

        // Check for errors in response
        CheckAuditLogResponse(response, entry.Field);
    }

    private void CheckAuditLogResponse(string response, string field)
    {
        var app = ctx.App;
        if (app == null) return;

        try
        {
            using var doc = JsonDocument.Parse(response);
            if (doc.RootElement.TryGetProperty("error", out var errorEl))
            {
                var errorCode = errorEl.GetString();
                app.Logger.LogWarning("Audit log entry failed for field {Field}: {Error}", field, errorCode);
            }
            else if (doc.RootElement.TryGetProperty("auditId", out var auditIdEl))
            {
                app.Logger.LogDebug("Audit entry created successfully: {AuditId}", auditIdEl.GetString());
            }
        }
        catch (Exception ex)
        {
            app.Logger.LogTrace(ex, "Could not parse audit log response (non-critical)");
        }
    }

    private string BuildAuditStatement(AuditLogEntry entry, SupplementaryAreasPlugin.Person? currentUser)
    {
        var userName = currentUser?.DisplayName;

        // If area-specific change
        if (entry.AreaIndex.HasValue)
        {
            var areaInfo = string.IsNullOrWhiteSpace(entry.AreaLabel)
                ? $"Supplementary Area {entry.AreaIndex}"
                : $"Supplementary Area {entry.AreaIndex} ({entry.AreaLabel})";

            return $"{userName} updated the {areaInfo}";
        }

        // Global change (like DueDate)
        return $"{userName} updated the Supplementary Areas";
    }

    private static string? ValidateAndTruncate(string? value, int max)
        => string.IsNullOrEmpty(value) ? value : (value.Length <= max ? value : value.Substring(0, max));

    /// <summary>
    /// Parse date string and convert to ISO 8601 format string (UTC) for consistency with system date storage
    /// </summary>
    /// <param name="dateString">Date string in various formats (e.g., "2025-10-16" or "2025-10-16T00:00:00")</param>
    /// <returns>ISO 8601 formatted date string or null if parsing fails</returns>
    private static string? ParseAndFormatDateString(string? dateString)
    {
        if (string.IsNullOrWhiteSpace(dateString))
            return null;

        try
        {
            // Try to parse the date string
            if (DateTime.TryParse(dateString, null, System.Globalization.DateTimeStyles.RoundtripKind, out var date))
            {
                // Convert to UTC if not already
                if (date.Kind == DateTimeKind.Unspecified)
                {
                    // Treat unspecified dates as UTC
                    date = DateTime.SpecifyKind(date, DateTimeKind.Utc);
                }
                else if (date.Kind == DateTimeKind.Local)
                {
                    date = date.ToUniversalTime();
                }

                // Return in ISO 8601 format: 2025-10-23T04:00:00Z
                return date.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }

            return null;
        }
        catch
        {
            return null;
        }
    }

    private static SupplementaryAreasPlugin.SupplementaryArea CreateNewArea(int index) => new()
    {
        Id = Guid.NewGuid().ToString("n"),
        Index = index,
        Order = index,
        Label = null,
        ExecutiveDirectors = new List<SupplementaryAreasPlugin.Person>(),
        Allocators = new List<SupplementaryAreasPlugin.Person>(),
        Contributors = new List<SupplementaryAreasPlugin.Person>(),
        Comments = new List<SupplementaryAreasPlugin.Comment>(),
        EndorsementComplete = false
    };

    private sealed class CompleteStatePayload
    {
        public int Version { get; set; }
        public string? DueDate { get; set; }
        public List<SupplementaryAreasPlugin.SupplementaryArea> Areas { get; set; } = new();
    }

    private sealed class FieldDescriptorLite
    {
        public string InternalName { get; set; } = string.Empty;
    }
}
