using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using DPC.APC.Plugins.SDK;
using Microsoft.Extensions.Logging;

namespace SupplementaryAreaPlugin.Actions;

/// <summary>
/// Tracks changes between old and new supplementary area models
/// </summary>
internal sealed class SupplementaryChangeTracker
{
    private readonly SupplementaryAreasPlugin.SupplementaryModel oldModel;
    private readonly SupplementaryAreasPlugin.SupplementaryModel newModel;
    private readonly ILogger logger;

    public SupplementaryChangeTracker(
        SupplementaryAreasPlugin.SupplementaryModel oldModel,
        SupplementaryAreasPlugin.SupplementaryModel newModel,
        ILogger logger)
    {
        this.oldModel = oldModel ?? throw new ArgumentNullException(nameof(oldModel));
        this.newModel = newModel ?? throw new ArgumentNullException(nameof(newModel));
        this.logger = logger ?? throw new ArgumentNullException(nameof(logger));
    }

    /// <summary>
    /// Analyzes all changes between old and new models
    /// </summary>
    public ModelChanges DetectChanges()
    {
        var changes = new ModelChanges();

        // Track due date changes
        if (oldModel.DueDate != newModel.DueDate)
        {
            changes.DueDateChanged = true;
            changes.AuditLogEntries.Add(new AuditLogEntry
            {
                Field = "DueDate",
                OldValue = oldModel.DueDate,
                NewValue = newModel.DueDate
            });
        }

        // Track area-level changes
        foreach (var newArea in newModel.Areas)
        {
            var oldArea = oldModel.Areas?.FirstOrDefault(a => a.Id == newArea.Id);

            if (oldArea == null)
            {
                // New area added
                TrackNewArea(newArea, changes);
            }
            else
            {
                // Existing area - track changes
                TrackAreaChanges(oldArea, newArea, changes);
            }
        }

        // Track deleted areas
        if (oldModel.Areas != null)
        {
            foreach (var oldArea in oldModel.Areas)
            {
                if (newModel.Areas.All(a => a.Id != oldArea.Id))
                {
                    changes.AuditLogEntries.Add(new AuditLogEntry
                    {
                        Field = $"Supplementary Area {oldArea.Index}",
                        OldValue = oldArea.Label,
                        NewValue = null,
                        AreaIndex = oldArea.Index,
                        AreaLabel = oldArea.Label
                    });
                }
            }
        }

        logger.LogDebug(
            "Change tracking: {NewPeopleCount} new people, {EndorsementCount} endorsements completed, {AuditCount} audit entries",
            changes.NewPeopleAssignments.Count,
            changes.EndorsementCompletions.Count,
            changes.AuditLogEntries.Count);

        return changes;
    }

    private void TrackNewArea(SupplementaryAreasPlugin.SupplementaryArea newArea, ModelChanges changes)
    {
        changes.AuditLogEntries.Add(new AuditLogEntry
        {
            Field = $"Supplementary Area {newArea.Index}",
            OldValue = null,
            NewValue = newArea.Label,
            AreaIndex = newArea.Index,
            AreaLabel = newArea.Label
        });

        // All people in new area are "new assignments"
        TrackNewPeopleInArea(null, newArea, changes);
    }

    private void TrackAreaChanges(
        SupplementaryAreasPlugin.SupplementaryArea oldArea,
        SupplementaryAreasPlugin.SupplementaryArea newArea,
        ModelChanges changes)
    {
        var areaIndex = newArea.Index;
        var areaLabel = newArea.Label;

        // Track label changes
        if (oldArea.Label != newArea.Label)
        {
            changes.AuditLogEntries.Add(new AuditLogEntry
            {
                Field = "Name",
                OldValue = oldArea.Label,
                NewValue = newArea.Label,
                AreaIndex = areaIndex,
                AreaLabel = areaLabel
            });
        }

        // Track endorsement completion
        if (!oldArea.EndorsementComplete && newArea.EndorsementComplete)
        {
            changes.EndorsementCompletions.Add(new EndorsementCompletion
            {
                AreaIndex = areaIndex,
                AreaLabel = areaLabel,
            });

            changes.AuditLogEntries.Add(new AuditLogEntry
            {
                Field = $"Endorsement",
                OldValue = "Incomplete",
                NewValue = "Complete",
                AreaIndex = areaIndex,
                AreaLabel = areaLabel
            });

            // Add to key details changes for notification
            var key = FormatKeyDetailsKey(areaIndex, areaLabel, "Endorsed");
            changes.KeyDetailsChanges[key] = "Complete";
        }
        else if (oldArea.EndorsementComplete && !newArea.EndorsementComplete)
        {
            // Endorsement uncompleted
            changes.AuditLogEntries.Add(new AuditLogEntry
            {
                Field = $"Endorsement",
                OldValue = "Complete",
                NewValue = "Incomplete",
                AreaIndex = areaIndex,
                AreaLabel = areaLabel
            });

            // Add to key details changes for notification
            var key = FormatKeyDetailsKey(areaIndex, areaLabel, "Endorsed");
            changes.KeyDetailsChanges[key] = "Incomplete";
        }

        // Track people changes for each role
        TrackPeopleChanges("Executive Directors", oldArea.ExecutiveDirectors, newArea.ExecutiveDirectors, areaIndex, areaLabel, changes);
        TrackPeopleChanges("Allocators", oldArea.Allocators, newArea.Allocators, areaIndex, areaLabel, changes);
        TrackPeopleChanges("Contributors", oldArea.Contributors, newArea.Contributors, areaIndex, areaLabel, changes);

        // Track new people assignments
        TrackNewPeopleInArea(oldArea, newArea, changes);

        // Track comment additions
        var oldCommentCount = oldArea.Comments?.Count ?? 0;
        var newCommentCount = newArea.Comments?.Count ?? 0;
        if (newCommentCount > oldCommentCount)
        {
            changes.AuditLogEntries.Add(new AuditLogEntry
            {
                Field = "Comments",
                OldValue = $"{oldCommentCount} comment(s)",
                NewValue = $"{newCommentCount} comment(s)",
                AreaIndex = areaIndex,
                AreaLabel = areaLabel
            });

            // Track new comments for key details notification
            var oldCommentIds = new HashSet<string>(
                oldArea.Comments?.Select(c => c.Id) ?? Enumerable.Empty<string>(),
                StringComparer.OrdinalIgnoreCase);
            
            var newComments = newArea.Comments?.Where(c => !oldCommentIds.Contains(c.Id)).ToList() 
                ?? new List<SupplementaryAreasPlugin.Comment>();
            
            foreach (var comment in newComments)
            {
                var key = FormatKeyDetailsKey(areaIndex, areaLabel, "Comment");
                var commentPreview = comment.Text?.Length > 100 
                    ? comment.Text.Substring(0, 100) + "..." 
                    : comment.Text ?? string.Empty;
                changes.KeyDetailsChanges[key] = commentPreview;
            }
        }
    }

    private void TrackPeopleChanges(
        string roleTitle,
        List<SupplementaryAreasPlugin.Person>? oldPeople,
        List<SupplementaryAreasPlugin.Person>? newPeople,
        int areaIndex,
        string? areaLabel,
        ModelChanges changes)
    {
        var oldIds = GetPersonIds(oldPeople);
        var newIds = GetPersonIds(newPeople);

        var added = newIds.Except(oldIds).ToList();
        var removed = oldIds.Except(newIds).ToList();

        if (added.Count > 0 || removed.Count > 0)
        {
            var oldDisplay = FormatPeopleList(oldPeople);
            var newDisplay = FormatPeopleList(newPeople);

            changes.AuditLogEntries.Add(new AuditLogEntry
            {
                Field = $"{roleTitle}",
                OldValue = oldDisplay,
                NewValue = newDisplay,
                AreaIndex = areaIndex,
                AreaLabel = areaLabel
            });
        }
    }

    private void TrackNewPeopleInArea(
        SupplementaryAreasPlugin.SupplementaryArea? oldArea,
        SupplementaryAreasPlugin.SupplementaryArea newArea,
        ModelChanges changes)
    {
        TrackNewPeopleForRole("ExecutiveDirectors", "Executive Directors", oldArea?.ExecutiveDirectors, newArea.ExecutiveDirectors, newArea, changes, NotificationType.LeaderAssigned);
        TrackNewPeopleForRole("Allocators", "Allocators", oldArea?.Allocators, newArea.Allocators, newArea, changes, NotificationType.LeaderAssigned);
        TrackNewPeopleForRole("Contributors", "Contributors", oldArea?.Contributors, newArea.Contributors, newArea, changes, NotificationType.ContributorAssigned);
    }

    private void TrackNewPeopleForRole(
        string roleField,
        string roleDisplayName,
        List<SupplementaryAreasPlugin.Person>? oldPeople,
        List<SupplementaryAreasPlugin.Person>? newPeople,
        SupplementaryAreasPlugin.SupplementaryArea area,
        ModelChanges changes,
        NotificationType notificationType)
    {
        var oldIds = GetPersonIds(oldPeople);
        var newPeopleList = newPeople ?? new List<SupplementaryAreasPlugin.Person>();

        foreach (var person in newPeopleList)
        {
            var personId = GetPersonId(person);
            if (string.IsNullOrWhiteSpace(personId)) continue;
            
            if (!oldIds.Contains(personId))
            {
                changes.NewPeopleAssignments.Add(new PersonAssignment
                {
                    Person = person,
                    Role = roleDisplayName,
                    RoleField = roleField,
                    AreaIndex = area.Index,
                    AreaLabel = area.Label,
                    NotificationType = notificationType
                });
            }
        }
    }

    private HashSet<string> GetPersonIds(List<SupplementaryAreasPlugin.Person>? people)
    {
        if (people == null || people.Count == 0)
        {
            return new HashSet<string>(StringComparer.OrdinalIgnoreCase);
        }

        var result = new HashSet<string>(
            people.Select(GetPersonId).Where(id => !string.IsNullOrWhiteSpace(id))!,
            StringComparer.OrdinalIgnoreCase);
        
        return result;
    }

    private string? GetPersonId(SupplementaryAreasPlugin.Person person)
    {
        if (person == null)
        {
            return null;
        }
        return person.GraphUserId ?? person.UserPrincipalName ?? person.Mail;
    }

    private string FormatPeopleList(List<SupplementaryAreasPlugin.Person>? people)
    {
        if (people == null || people.Count == 0)
        {
            return null;
        }

        var names = people
            .Select(p => p.DisplayName ?? p.UserPrincipalName ?? p.Mail ?? "Unknown")
            .ToList();

        var result = string.Join(", ", names);
        return result;
    }

    /// <summary>
    /// Formats a key for KeyDetailsChanges dictionary
    /// Format: "Supplementary Area {AreaIndex} ({AreaLabel}): {fieldName}"
    /// </summary>
    private string FormatKeyDetailsKey(int areaIndex, string? areaLabel, string fieldName)
    {
        if (string.IsNullOrWhiteSpace(areaLabel))
        {
            return $"Supplementary Area {areaIndex}: {fieldName}";
        }
        return $"Supplementary Area {areaIndex} ({areaLabel}): {fieldName}";
    }
}

/// <summary>
/// Contains all detected changes between old and new models
/// </summary>
internal sealed class ModelChanges
{
    public bool DueDateChanged { get; set; }
    public List<PersonAssignment> NewPeopleAssignments { get; set; } = new();
    public List<EndorsementCompletion> EndorsementCompletions { get; set; } = new();
    public List<AuditLogEntry> AuditLogEntries { get; set; } = new();
    /// <summary>
    /// Key details changes for SendKeyDetailsChangedNotification
    /// Key format: "Supplementary Area {AreaIndex} ({AreaLabel}): {Endorsed/Comment}"
    /// Value: the new field value
    /// </summary>
    public Dictionary<string, string> KeyDetailsChanges { get; set; } = new();
}

/// <summary>
/// Represents a new person assigned to an area
/// </summary>
internal sealed class PersonAssignment
{
    public SupplementaryAreasPlugin.Person Person { get; set; } = null!;
    public string Role { get; set; } = string.Empty;
    public string RoleField { get; set; } = string.Empty;
    public int AreaIndex { get; set; }
    public string AreaLabel { get; set; } = string.Empty;
    public NotificationType NotificationType { get; set; } = NotificationType.LeaderAssigned;
}

/// <summary>
/// Notification types for supplementary area changes
/// </summary>
internal enum NotificationType
{
    LeaderAssigned,      // Executive Director or Allocator
    ContributorAssigned, // Contributor
    EndorsementCompleted // Endorsement marked complete
}

/// <summary>
/// Represents an area where endorsement was completed
/// </summary>
internal sealed class EndorsementCompletion
{
    public int AreaIndex { get; set; }
    public string AreaLabel { get; set; } = string.Empty;
}

/// <summary>
/// Represents an audit log entry for a field change
/// </summary>
internal sealed class AuditLogEntry
{
    public string Field { get; set; } = string.Empty;
    public string? OldValue { get; set; } = string.Empty;
    public string? NewValue { get; set; } = string.Empty;
    public int? AreaIndex { get; set; }
    public string? AreaLabel { get; set; }
}

